<?php
/**
 * Main plugin class
 */
class Nietos_AI {

    /**
     * Plugin version
     */
    const VERSION = NIETOS_AI_VERSION;

    /**
     * Security instance
     */
    private $security;

    /**
     * Settings instance
     */
    private $settings;

    /**
     * API handler instance
     */
    private $api_handler;

    /**
     * REST controller instance
     */
    private $rest_controller;

    /**
     * Admin page instance
     */
    private $admin_page;

    /**
     * Editor integration instance
     */
    private $editor_integration;

    /**
     * Constructor
     */
    public function __construct() {
        $this->security = new Nietos_AI_Security();
        $this->settings = new Nietos_AI_Settings();
        $this->api_handler = new Nietos_AI_API_Handler($this->security);
        $this->rest_controller = new Nietos_AI_REST_Controller($this->api_handler, $this->settings);
        $this->admin_page = new Nietos_AI_Admin_Page($this->settings);
        $this->editor_integration = new Nietos_AI_Editor_Integration();
    }

    /**
     * Run the plugin
     */
    public function run() {
        // Initialize REST API
        add_action('rest_api_init', array($this->rest_controller, 'register_routes'));

        // Initialize admin
        if (is_admin()) {
            add_action('admin_menu', array($this->admin_page, 'add_admin_menu'));
            add_action('admin_init', array($this->settings, 'register_settings'));
            add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        }

        // Initialize editor integration
        add_action('enqueue_block_editor_assets', array($this->editor_integration, 'enqueue_editor_assets'));

        // Increase timeout for OpenAI API requests
        add_filter('http_request_timeout', array($this, 'increase_openai_timeout'), 10, 2);
    }

    /**
     * Increase timeout for OpenAI API requests
     */
    public function increase_openai_timeout($timeout, $url) {
        // Only increase timeout for OpenAI API requests
        if (strpos($url, 'api.openai.com') !== false) {
            return 120; // 120 seconds for OpenAI requests
        }
        return $timeout;
    }

    /**
     * Enqueue admin scripts and styles
     */
    public function enqueue_admin_scripts($hook) {
        // Only load on our admin page
        if ($hook !== 'toplevel_page_nietos-ai') {
            return;
        }

        // Enqueue React app build for chat interface
        $main_js = NIETOS_AI_PLUGIN_DIR . 'dist/assets/main.js';

        if (file_exists($main_js)) {
            // Localize script with necessary data (must be before enqueue)
            wp_register_script('nietos-ai-admin', NIETOS_AI_PLUGIN_URL . 'dist/assets/main.js', array(), self::VERSION, true);
            wp_localize_script('nietos-ai-admin', 'nietosAI', array(
                'ajaxUrl' => admin_url('admin-ajax.php'),
                'restUrl' => rest_url('nietos-ai/v1'),
                'nonce' => wp_create_nonce('wp_rest'),
                'userId' => get_current_user_id(),
                'siteUrl' => get_site_url(),
                'isConfigured' => $this->settings->is_configured()
            ));

            // Enqueue the script
            wp_enqueue_script('nietos-ai-admin');

            // Add module type attribute for ES modules
            add_filter('script_loader_tag', array($this, 'add_type_attribute'), 10, 3);
        }
    }

    /**
     * Add type="module" attribute to our scripts
     */
    public function add_type_attribute($tag, $handle, $src) {
        // Add module type to our scripts
        if ($handle === 'nietos-ai-admin') {
            $tag = '<script type="module" src="' . esc_url($src) . '" id="' . $handle . '-js"></script>';
        }
        return $tag;
    }
}
