<?php
/**
 * Premium features management
 * Handles message limits and premium feature access
 */
class Nietos_AI_Premium {

    /**
     * Free version message limit
     */
    const FREE_MESSAGE_LIMIT = 5;

    /**
     * Message count option (per session)
     */
    const MESSAGE_COUNT_OPTION = 'nietos_ai_message_count';

    /**
     * License instance
     */
    private $license;

    /**
     * Constructor
     */
    public function __construct() {
        $this->license = new Nietos_AI_License();
    }

    /**
     * Check if user has premium license
     */
    public function has_premium_license() {
        return $this->license->is_premium_active();
    }

    /**
     * Check if user can send messages (within limit)
     */
    public function can_send_message() {
        // Premium users have unlimited messages
        if ($this->has_premium_license()) {
            return true;
        }

        // Check free user message count
        $count = $this->get_message_count();

        return $count < self::FREE_MESSAGE_LIMIT;
    }

    /**
     * Get current message count for this session
     */
    public function get_message_count() {
        $user_id = get_current_user_id();
        $count_data = get_user_meta($user_id, self::MESSAGE_COUNT_OPTION, true);

        if (empty($count_data) || !is_array($count_data)) {
            return 0;
        }

        // Reset count if it's a new day
        $today = current_time('Y-m-d');
        if (!isset($count_data['date']) || $count_data['date'] !== $today) {
            return 0;
        }

        return isset($count_data['count']) ? intval($count_data['count']) : 0;
    }

    /**
     * Increment message count
     */
    public function increment_message_count() {
        // Premium users don't need counting
        if ($this->has_premium_license()) {
            return;
        }

        $user_id = get_current_user_id();
        $today = current_time('Y-m-d');

        $count_data = array(
            'date' => $today,
            'count' => $this->get_message_count() + 1
        );

        update_user_meta($user_id, self::MESSAGE_COUNT_OPTION, $count_data);
    }

    /**
     * Get remaining messages for free users
     */
    public function get_remaining_messages() {
        if ($this->has_premium_license()) {
            return -1; // Unlimited
        }

        $count = $this->get_message_count();
        return max(0, self::FREE_MESSAGE_LIMIT - $count);
    }

    /**
     * Reset message count (for testing or admin purposes)
     */
    public function reset_message_count() {
        $user_id = get_current_user_id();
        delete_user_meta($user_id, self::MESSAGE_COUNT_OPTION);
    }

    /**
     * Save license key (delegates to License class)
     */
    public function save_license($license_key) {
        $result = $this->license->activate_license($license_key);
        return $result['success'] ?? false;
    }

    /**
     * Remove license (delegates to License class)
     */
    public function remove_license() {
        $this->license->deactivate_license();
    }

    /**
     * Get full license key (delegates to License class)
     */
    public function get_license_key() {
        return $this->license->get_license_key();
    }

    /**
     * Get license key masked for display
     */
    public function get_license_key_masked() {
        $license = $this->license->get_license_key();

        if (empty($license)) {
            return '';
        }

        // Show only first 4 and last 4 characters
        if (strlen($license) > 12) {
            return substr($license, 0, 4) . str_repeat('*', strlen($license) - 8) . substr($license, -4);
        }

        return str_repeat('*', strlen($license));
    }
}
