# Create Operation Examples

These examples demonstrate the correct way to handle CREATE operations.

---

## ⚠️ CRITICAL: Always Analyze Existing Content First

**Before creating ANY new post or page, you MUST:**
1. Call `find_posts()` to get 2-3 recent similar posts
2. Call `get_post_blocks()` on those posts to see their structure
3. Match the structure exactly in your new content

**Why?** Sites may use:
- Gutenberg blocks (with proper `<!-- wp:heading -->` blocks)
- Classic Editor (plain HTML)
- ACF custom blocks
- Theme-specific blocks

**❌ NEVER:**
- Use markdown headings (`##`) in Gutenberg sites
- Mix Classic Editor with Gutenberg blocks
- Assume the structure without checking
- Create inconsistent content

---

## Simple Creates

### Example 1: Create Blog Post (Correct Process)
```
User: "Create a new blog post about WordPress security"

AI Step 1: "Let me first check how your existing posts are structured..."
AI Action: find_posts(post_type: 'post', number: 3)
AI Action: get_post_blocks(post_id: [recent post ID])

AI Step 2: Analyze structure → "I see your posts use Gutenberg blocks with proper heading blocks."

AI Step 3: "I'll create a new blog post about WordPress security matching your existing post structure."
AI Action: create_post({
  title: "WordPress Security Best Practices",
  content: "<!-- wp:paragraph --><p>Website security is crucial...</p><!-- /wp:paragraph -->
            <!-- wp:heading {\"level\":2} --><h2>Essential Security Measures</h2><!-- /wp:heading -->
            <!-- wp:paragraph --><p>Here are the key practices...</p><!-- /wp:paragraph -->
            <!-- wp:heading {\"level\":3} --><h3>Use Strong Passwords</h3><!-- /wp:heading -->
            <!-- wp:paragraph --><p>Always use complex passwords...</p><!-- /wp:paragraph -->",
  post_type: "post",
  status: "draft"
})
AI Confirmation: "Done! I've created a new draft post titled 'WordPress Security Best Practices' using the same Gutenberg block structure as your existing posts. Edit it here: [edit_url]"
```

### Example 2: Create Page
```
User: "Make a contact page"

AI Response: "I'll create a new contact page with a contact form."
AI Action: create_post({
  title: "Contact Us",
  content: "[contact form block + intro text]",
  post_type: "page",
  status: "draft"
})
AI Confirmation: "Created a new 'Contact Us' page with a contact form. It's currently a draft. Edit it here: [edit_url]"
```

### Example 3: Create with Specific Content
```
User: "Create a post called 'Getting Started' with an introduction and 3 steps"

AI Response: "I'll create a 'Getting Started' post with an introduction and 3 steps."
AI Action: create_post({
  title: "Getting Started",
  content: "<!-- wp:paragraph --><p>Welcome! Here's how to get started:</p><!-- /wp:paragraph -->
            <!-- wp:heading --><h2>Step 1: [title]</h2><!-- /wp:heading -->
            <!-- wp:paragraph --><p>[content]</p><!-- /wp:paragraph -->
            [... steps 2 and 3 ...]",
  status: "draft"
})
AI Confirmation: "Created 'Getting Started' post with introduction and 3 steps. Review it here: [edit_url]"
```

---

## Creates with Metadata

### Example 4: Create with Category
```
User: "Create a new tutorial post in the 'WordPress' category"

AI Response: "I'll create a new tutorial post and add it to the 'WordPress' category."
AI Action:
  list_terms(taxonomy: 'category') → find WordPress category ID
  create_post({
    title: "New WordPress Tutorial",
    content: "[tutorial content]",
    post_type: "post",
    status: "draft"
  }) → get new post_id
  assign_terms_to_post(post_id, taxonomy: 'category', term_ids: [wordpress_category_id])
AI Confirmation: "Created new tutorial post in the 'WordPress' category. Edit here: [edit_url]"
```

### Example 5: Create with Tags
```
User: "Make a post about CSS tips with tags 'css', 'design', 'frontend'"

AI Response: "I'll create a post about CSS tips with the specified tags."
AI Action:
  create_post({
    title: "Essential CSS Tips",
    content: "[CSS tips content]",
    status: "draft"
  }) → get new post_id
  list_terms(taxonomy: 'post_tag') → find or create tag IDs
  assign_terms_to_post(post_id, taxonomy: 'post_tag', term_ids: [css_id, design_id, frontend_id])
AI Confirmation: "Created post with CSS, design, and frontend tags. Edit here: [edit_url]"
```

---

## Creates Requiring Clarification

### Example 6: Insufficient Information
```
User: "Create a new page"

AI Response: "I can create a new page for you. What should it be called, and what content would you like on it?"

User: "Call it 'Services' and list our web design and development services"

AI Action: create_post({
  title: "Services",
  content: "[generated content about web design and development services]",
  post_type: "page",
  status: "draft"
})
```

### Example 7: Clarify Post vs Page
```
User: "Create something about our team"

AI Response: "Should this be a blog post or a static page?"

User: "A page"

AI Action: create_post({
  title: "Our Team",
  content: "[team page content]",
  post_type: "page",
  status: "draft"
})
```

---

## What NOT to Do

### WRONG Example 1: Editing Instead of Creating
```
User: "Create a new post about React"
Context: Currently editing post ID 123

❌ WRONG AI Action: update_post(123, { title: "About React" })
✅ CORRECT AI Action: create_post({ title: "Getting Started with React", content: "..." })
```

### WRONG Example 2: Publishing Without Confirmation
```
User: "Create a new blog post"

❌ WRONG AI Action: create_post({ ..., status: "publish" })
✅ CORRECT AI Action: create_post({ ..., status: "draft" })

Reason: Always default to 'draft' unless user explicitly asks to publish
```

### WRONG Example 3: No Content Generation
```
User: "Create a post about WordPress plugins"

❌ WRONG AI Action: create_post({ title: "WordPress Plugins", content: "" })
✅ CORRECT AI Action: create_post({
  title: "Essential WordPress Plugins",
  content: "[generated introduction, plugin recommendations with descriptions]"
})

Reason: Generate helpful starter content, not empty posts
```

---

## Advanced Creates

### Example 8: Create with Featured Image
```
User: "Create a blog post about summer with image #456 as featured image"

AI Action:
  create_post({
    title: "Summer Activities",
    content: "[summer content]",
    status: "draft"
  }) → get new post_id
  set_featured_image(post_id, attachment_id: 456)
AI Confirmation: "Created 'Summer Activities' post with featured image. Edit here: [edit_url]"
```

### Example 9: Create Scheduled Post
```
User: "Create a post about the upcoming event, scheduled for next Monday"

AI Response: "I'll create a post scheduled to publish next Monday at 9:00 AM."
AI Action: create_post({
  title: "Upcoming Event",
  content: "[event details]",
  status: "future",
  date: "2024-MM-DD 09:00:00"
})
AI Confirmation: "Created post scheduled for Monday, [date] at 9:00 AM."
```

---

## Batch Creates

### Example 10: Create Multiple Posts
```
User: "Create 3 blog posts: one about React, one about Vue, one about Angular"

AI Response: "I'll create 3 draft blog posts about React, Vue, and Angular."
AI Actions:
  create_post({ title: "Getting Started with React", content: "..." })
  create_post({ title: "Vue.js for Beginners", content: "..." })
  create_post({ title: "Angular Framework Guide", content: "..." })
AI Confirmation: "Created 3 posts:
1. Getting Started with React - Edit: [url1]
2. Vue.js for Beginners - Edit: [url2]
3. Angular Framework Guide - Edit: [url3]"
```

---

## Content Generation Best Practices

### Good Content Structure for Blog Posts:
```
- Introduction paragraph (what is this post about?)
- 2-3 main sections with headings
- Bullet points or numbered lists where appropriate
- Conclusion or call-to-action
- Use appropriate Gutenberg blocks
```

### Good Content Structure for Pages:
```
- Welcome/intro paragraph
- Clear sections with headings
- Use of columns, buttons, or other page elements as appropriate
- Contact information or forms if relevant
- Clean, professional layout
```

---

## Key Takeaways

1. ✅ **ALWAYS** use create_post() for new content, never update_post()
2. ✅ **DEFAULT** to 'draft' status unless explicitly asked to publish
3. ✅ **GENERATE** helpful starter content, not empty posts
4. ✅ **ASK** for required information if not provided
5. ✅ **PROVIDE** edit links after creation
6. ✅ **DISTINGUISH** between posts and pages clearly
7. ✅ **SET** appropriate metadata (categories, tags) when specified
