# Edit Operation Examples

These examples demonstrate the correct way to handle EDIT operations.

---

## Simple Edits

### Example 1: Edit Title
```
User: "Edit this page's title to 'About Us'"
Context: Currently editing page ID 42, title is "About"

AI Response: "I'll update the page title to 'About Us'."
AI Action: update_post(42, { title: "About Us" })
AI Confirmation: "Done! Page title updated to 'About Us'."
```

### Example 2: Change Excerpt
```
User: "Change the excerpt"
Context: Currently editing post ID 123

AI Response: "What would you like the new excerpt to be?"
User: "This post explains WordPress security best practices"
AI Action: update_post(123, { excerpt: "This post explains WordPress security best practices" })
AI Confirmation: "Excerpt updated successfully!"
```

### Example 3: Update Content
```
User: "Add a paragraph about mobile responsiveness"
Context: Currently editing post ID 456

AI Response: "I'll add a paragraph about mobile responsiveness to your post."
AI Action:
  get_post_blocks(456) → retrieve current blocks
  update_post_blocks(456, { mode: 'append', blocks: [{ type: "paragraph", content: "[generated content about mobile responsiveness]" }] })
AI Confirmation: "Added paragraph about mobile responsiveness to the post."
```

---

## Block Editing

### Example 4: Modify Block Content
```
User: "Change the heading to say 'Welcome to Our Site'"
Context: Currently editing post ID 123

AI Response: "I'll change the heading text."
AI Action:
  get_post_blocks(123) → retrieve current blocks
  update_post_blocks(123, { gutenberg_content: "<!-- wp:heading --><h2>Welcome to Our Site</h2><!-- /wp:heading -->..." })
AI Confirmation: "Heading text updated."
```

### Example 5: Change Image
```
User: "Replace the featured image with image #789"
Context: Currently editing post ID 234

AI Response: "I'll set image #789 as the featured image."
AI Action: update_post(234, { featured_media: 789 })
AI Confirmation: "Featured image updated!"
```

---

## Ambiguous Requests (Require Clarification)

### Example 6: Unclear Target
```
User: "Edit the blog post about WordPress"
Context: Admin dashboard (no specific post in context)

AI Response: "I found 3 posts about WordPress:
1. 'WordPress Security Tips' (ID: 123, Published)
2. 'Getting Started with WordPress' (ID: 456, Draft)
3. 'WordPress vs Other CMS' (ID: 789, Published)

Which one would you like to edit?"
```

### Example 7: Vague Edit Request
```
User: "Fix the post"
Context: Currently editing post ID 111

AI Response: "I can help fix the post. What would you like me to fix? For example:
- Fix spelling/grammar
- Fix formatting issues
- Fix broken links
- Something else?"
```

---

## What NOT to Do

### WRONG Example 1: Creating Instead of Editing
```
User: "Edit this page's title to 'Contact'"
Context: Currently editing page ID 42

❌ WRONG AI Action: create_post({ title: "Contact", post_type: "page" })
✅ CORRECT AI Action: update_post(42, { title: "Contact" })
```

### WRONG Example 2: Replacing All Content
```
User: "Add a conclusion paragraph"
Context: Post has 5 paragraphs of existing content

❌ WRONG: Replace all content with just the conclusion
✅ CORRECT: Append conclusion paragraph to existing content
```

### WRONG Example 3: Not Using Context
```
User: "Change the title to 'New Title'"
Context: Currently editing post ID 999

❌ WRONG: Ask "Which post do you want to edit?"
✅ CORRECT: Update the current post (ID 999) title
```

---

## Metadata Edits

### Example 8: Change Categories
```
User: "Add this post to the 'Tutorials' category"
Context: Post ID 555, currently in 'General' category

AI Response: "I'll add the 'Tutorials' category while keeping the existing 'General' category."
AI Action:
  list_terms(taxonomy: 'category') → find Tutorials category ID
  assign_terms_to_post(555, taxonomy: 'category', term_ids: [general_id, tutorials_id], append: false)
AI Confirmation: "Post is now in both 'General' and 'Tutorials' categories."
```

### Example 9: Update Status
```
User: "Publish this post"
Context: Post ID 666, currently 'draft' status

AI Response: "This post is currently a draft. Are you ready to publish it? Once published, it will be visible to all site visitors."
User: "Yes"
AI Action: update_post(666, { status: "publish" })
AI Confirmation: "Post published successfully! View it here: [post_url]"
```

---

## Batch Edits

### Example 10: Edit Multiple Fields
```
User: "Change the title to 'New Title' and add it to the 'Featured' category"
Context: Post ID 777

AI Response: "I'll update the title and add the 'Featured' category."
AI Action:
  update_post(777, { title: "New Title" })
  list_terms(taxonomy: 'category') → find Featured category ID
  assign_terms_to_post(777, taxonomy: 'category', term_ids: [existing_ids, featured_id], append: false)
AI Confirmation: "Updated! Title changed to 'New Title' and added to 'Featured' category."
```

---

## Key Takeaways

1. ✅ **ALWAYS** use update_post() for edits, never create_post()
2. ✅ **USE CONTEXT** when available (current post ID in editor)
3. ✅ **ASK** when target is unclear
4. ✅ **PRESERVE** existing content unless removal is explicitly requested
5. ✅ **CONFIRM** destructive changes (status, deletions)
6. ✅ **UPDATE** only requested fields, leave others unchanged
